module net.BurtonRadons.dig.common.imageLoader;

private import net.BurtonRadons.dig.platform.base;
//private import std.stream;


/** An abstract interface for loading images.  You subclass this to be
  * served an image.  The only methods you certainly should handle
  * are dimensions and row; dimensions is called with the image data
  * (be sure to call the superclass), and row is called for each row in
  * the image.  To get row data in a format more to your liking, use
  * the convert method.
  */

class ImageFile
{
    /** Load image from the file. */
    this (char [] filename)
    {
        this (filename, new std.stream.File (filename));
    }

    /** Load image from the stream. */
    this (char [] filename, std.stream.Stream stream)
    {
        ImageLoader match = ImageLoader.detect (filename, stream);

        if (match === null)
            throw new Error ("Cannot find image loader for '" ~ filename ~ "'.");
        match.image = this;
        match.load ();
    }

    int width; /**< Width of the image, passed to dimensions. */
    int height; /**< Height of the image, passed to dimensions. */
    int depth; /**< Bit depth of each channel in the image, passed to dimensions.  This can be 1, 2, 4, 8, or 16. */
    char [] type; /**< Color type of the image; "luminance", "index", or "rgb". */
    bit alpha; /**< Whether this image has a dedicated alpha channel. */
    Color [] palette; /**< A previously given palette. */

    ubyte [] digCommonResult; /**< Used by convert. */

    /** The dimensions and color format of the subsequent image.
      * depth is the bit depth of each channel and can be 1, 2, 4,
      * 8, or 16.  type can be "luminance", "index", or "rgb".
      * alpha indicates whether an alpha channel is also given.
      */
    void dimensions (int width, int height, int depth, char [] type, bit alpha)
    {
        this.width = width;
        this.height = height;
        this.depth = depth;
        this.type = type;
        this.alpha = alpha;
    }

    /** Indicate that this image will be read in top-down row order.
      * The opposite of this is bottomup, although neither of these
      * will be called if it is unknowable.
      */
    void topdown ()
    {
    }

    /** Indicate that this image will be read in bottom-up row order. */
    void bottomup ()
    {
    }

    /** Indicate a palette. */
    void paletteColors (Color [] colors)
    {
        this.palette = colors;
    }

    /** Read in a std.math.single row.  y is the row index, and data is the
      * content.  For 1, 2, and 4-bit depth, the bits are in decreasing
      * order (the first channel of the first pixel of a bit image is
      * & 128 of the first byte, & 64 for the second channel, etcetera).
      * A type of luminance has a std.math.single channel, as does index, and
      * rgb has three in red, green, and blue order.  If there is an
      * alpha channel, this comes next.
      *
      * For example, take an rgb image with an alpha channel in 2-bit
      * depth.  The first byte can be taken apart as so:
      *
      * @code
      * ubyte v = first byte;
      * ubyte r = (v & 0b11000000) >> 6; // 0 to 3.
      * ubyte g = (v & 0b00110000) >> 4;
      * ubyte b = (v & 0b00001100) >> 2;
      * ubyte a = (v & 0b00000011);
      * @endcode
      *
      * There is no padding between pixels.
      */

    void row (int y, ubyte [] data)
    {
    }

    /** Return whether 16-bit images are okay.  You may still be served
      * them, in which case you should convert them to a lower depth,
      * but some packages can do this automatically.
      */
    bit prefer8to16 ()
    {
        return false;
    }

    /** Return whether 1, 2, and 4-bit images are okay.  You may still
      * be served them, in which case you should convert them to a higher
      * depth, but some packages can do this automatically.
      */
    bit prefer8to124 ()
    {
        return false;
    }

    /** Read an indexed value. */
    final int readIndex (ubyte [] data, int depth, int index)
    {
        if (depth == 1)
            return data [index / 8] & (1 << (7 - (index & 7))) ? 1 : 0;

        if (depth == 2)
        {
            ubyte v = data [index / 4];
            int s = (3 - (index & 3)) * 2;

            return (v & (3 << s)) >> s;
        }

        if (depth == 4)
        {
            ubyte v = data [index / 2];

            if (index & 1)
                return v & 0x0F;
            else
                return (v & 0xF0) >> 4;
        }

        if (depth == 8)
            return data [index];

        if (depth == 16)
            return data [index * 2 + 1] | (data [index * 2] << 8);
    }

    /** Read a value rescaled into another range. */
    final int readScale (ubyte [] data, int depth, int index, int maximum)
    {
        int value = readIndex (data, depth, index);

        switch (depth)
        {
            case 1: return value ? maximum : 0;
            case 2: return value * maximum / 3;
            case 4: return value * maximum / 15;
            case 8: return value * maximum / 255;
            case 16: return value * maximum / 65535;
        }
    }

    final void writeIndex (ubyte [] data, int depth, int index, uint value)
    {
        if (depth == 1)
        {
            ubyte *v = &data [index / 8];
            int i = 7 - (index & 7);

            *v = (*v & ~(1 << i)) | ((value & 1) << i);
        }
        else if (depth == 2)
        {
            ubyte *v = &data [index / 4];
            int i = (3 - (index & 3)) * 2;

            *v = (*v & ~(3 << i)) | ((value & 3) << i);
        }
        else if (depth == 4)
        {
            ubyte *v = &data [index / 2];
            int i = (1 - (index & 1)) * 4;

            *v = (*v & ~(15 << i)) | ((value & 15) << i);
        }
        else if (depth == 8)
            data [index] = value;
        else if (depth == 16)
        {
            data [index] = value >> 8;
            data [index + 1] = value;
        }
    }

    /** Write a value rescaled from another range. */
    final void writeScale (ubyte [] data, int depth, int index, uint value, int maximum)
    {
        switch (depth)
        {
            case 1: writeIndex (data, depth, index, value >= maximum / 2 ? 1 : 0); break;
            case 2: writeIndex (data, depth, index, value * 3 / maximum); break;
            case 4: writeIndex (data, depth, index, value * 15 / maximum); break;
            case 8: writeIndex (data, depth, index, value * 255 / maximum); break;
            case 16: writeIndex (data, depth, index, value * 65535 / maximum); break;
        }
    }

    /** Convert a row of data from the input to what you want. */
    ubyte [] convert (ubyte [] data, int ddepth, char [] dtype, bit dalpha)
    {
        if (ddepth == depth && dtype == type && dalpha == alpha)
            return data;

        int length = bytesPerRow (width, ddepth, dtype, dalpha);
        int srcBPP = bitsPerPixel ();
        int srcChannels = channelCount ();
        int dstChannels = channelCount (dtype, dalpha);

        if (digCommonResult.length < length)
        {
            delete digCommonResult;
            digCommonResult = new ubyte [length];
        }
        
        ubyte [] result = digCommonResult;

        for (int x; x < width; x ++)
        {
            int r, g, b, a, i, l;

            switch (depth)
            {
                case 8:
                    if (type == "rgb")
                    {
                        r = data [x * srcChannels + 0] * 65535 / 255;
                        g = data [x * srcChannels + 1] * 65535 / 255;
                        b = data [x * srcChannels + 2] * 65535 / 255;
                        if (alpha)
                            a = data [x * srcChannels + 3] * 65535 / 255;
                        else
                            a = 255;
                        l = (r / 256 * 2125 + g / 256 * 7154 + b / 256 * 721) / 10000 * 65535 / 255;
                        break;
                    }
                    else
                        goto srcDefault;

                default:
                srcDefault:
                    if (type == "index")
                    {
                        i = readIndex (data, depth, x * srcChannels);
                        r = (uint) palette [i].r * 65535 / 255;
                        g = (uint) palette [i].g * 65535 / 255;
                        b = (uint) palette [i].b * 65535 / 255;
                        a = (uint) palette [i].a * 65535 / 255;
                        l = (uint) palette [i].intensity () * 65535 / 255;

                        if (alpha)
                            a = readScale (data, depth, (x + 1) * srcChannels - 1, 65535);
                        else
                            a = 65535;
                    }
                    else if (type == "rgb")
                    {
                        Color color;

                        r = readScale (data, depth, x * srcChannels + 0, 65535);
                        g = readScale (data, depth, x * srcChannels + 1, 65535);
                        b = readScale (data, depth, x * srcChannels + 2, 65535);
                        if (alpha)
                            a = readScale (data, depth, x * srcChannels + 3, 65535);
                        else
                            a = 65535;
                        color = AColor (r / 256, g / 256, b / 256);
                        l = color.intensity () * 65535 / 255;
                    }
                    else
                        throw new Error (fmt ("Unhandled source depth %d, type %.*s, alpha %d", depth, type, alpha));
            }


            switch (depth)
            {
                case 8:
                    if (dtype == "rgb")
                    {
                        result [x * dstChannels + 0] = r * 255 / 65535;
                        result [x * dstChannels + 1] = g * 255 / 65535;
                        result [x * dstChannels + 2] = b * 255 / 65535;
                        if (dalpha)
                            result [x * dstChannels + 3] = a * 255 / 65535;
                    }
                    else
                        goto dstDefault;
                    break;

                default:
                dstDefault:
                    if (dtype == "rgb")
                    {
                        writeScale (result, ddepth, x * dstChannels + 0, r, 65535);
                        writeScale (result, ddepth, x * dstChannels + 1, g, 65535);
                        writeScale (result, ddepth, x * dstChannels + 2, b, 65535);
                    }
                    else if (dtype == "luminance")
                        writeScale (result, ddepth, x * dstChannels, l, 65535);
                    else
                        throw new Error (fmt ("Unhandled destination depth %d, type %.*s, alpha %d", ddepth, dtype, dalpha));

                    if (alpha)
                        writeScale (result, ddepth, (x + 1) * dstChannels - 1, a, 65535);
            }
        }

        return result;
    }

    /** Get the number of channels in a pixel specification. */
    int channelCount (char [] type, bit alpha)
    {
        switch (type)
        {
            case "luminance": return alpha ? 2 : 1;
            case "index": return alpha ? 2 : 1;
            case "rgb": return alpha ? 4 : 3; 
        }
    }

    /** Get the number of channels. */
    int channelCount ()
    {
        return channelCount (type, alpha);
    }

    /** Get the number of bits in a pixel. */
    int bitsPerPixel (int depth, char [] type, bit alpha)
    {
        return depth * channelCount (type, alpha);
    }

    /** Get the number of bits in a pixel. */
    int bitsPerPixel ()
    {
        return bitsPerPixel (depth, type, alpha);
    }

    /** Get the number of bytes in a row. */
    int bytesPerRow (int width, int depth, char [] type, bit alpha)
    {
        return (width * bitsPerPixel (depth, type, alpha) + 7) / 8;
    }

    /** Get the number of bytes in a row. */
    int bytesPerRow ()
    {
        return bytesPerRow (width, depth, type, alpha);
    }
}

/+
#ifdef DOXYGEN_SHOULD_BITE_ME
+/

private import net.BurtonRadons.dig.platform.control;
private import net.BurtonRadons.dig.platform.bitmap;
private import net.BurtonRadons.dig.platform.button;

private class digCommonBitmapImageFile : ImageFile
{
    Bitmap bitmap;

    this (char [] filename, std.stream.Stream stream)
    {
        super (filename, stream);
    }

    void dimensions (int width, int height, int depth, char [] type, bit alpha)
    {
        super.dimensions (width, height, depth, type, alpha);
        bitmap = new Bitmap (width, height);
    }

    void row (int y, ubyte [] data)
    {
        data = convert (data, 8, "rgb", true);
        bitmap.hseto (0, y, ((Color *) (void *) data) [0 .. width]);
    }

    bit prefer8to16 () { return true; }
    bit prefer8to124 () { return true; }
}

/+
#endif
+/

/** Read an image file and return a dig bitmap.
  * The only supported image formats at this time are png, jpeg, and tga.
  * @relates Bitmap
  */

Bitmap BitmapLoad (char [] filename)
{
    return (new digCommonBitmapImageFile (filename, new std.stream.File (filename))).bitmap;
}

/** Read an image file from a stream and return a bitmap.
  * filename will be used for error messages only.
  * @relates Bitmap
  */

Bitmap BitmapLoad (char [] filename, std.stream.Stream stream)
{
    return (new digCommonBitmapImageFile (filename, stream)).bitmap;
}

/** An image loader. */
class ImageLoader
{
    static ImageLoader [] digCommonList; /**< Various type loaders. */

    char [] filename; /**< Loaded by the file matcher. */
    std.stream.Stream stream; /**< Loaded by the file matcher. */
    ImageFile image; /**< Image file being loaded. */

    /** Register the loader. */
    this ()
    {
        digCommonList ~= this;
    }

    abstract char [] name (); /**< The type that this loads (for example, "PiNG"). */
    abstract char [] exts (); /**< Semicolon-separated, wildcard-using extensions that match this type. */
    abstract float match (); /**< Get the amount this stream matches the loader, from 0 (not at all) to 1 (completely). */
    abstract void load (); /**< Load the stream that you've previously identified. */

    /** Search through the registered loaders and return the best matching
      * loader for this file or null if there is none.
      */

    static ImageLoader detect (char[] filename, std.stream.Stream stream)
    {
        ImageLoader match;
        float best, compare;
        ulong tell;

        tell = stream.position ();

        for (int c; c < digCommonList.length; c ++)
        {
            ImageLoader loader = digCommonList [c];

            loader.filename = filename;
            loader.stream = stream;
            loader.image = null;
            compare = loader.match ();
            stream.position (tell);
            if (compare <= 0)
                continue;
            if (match === null || compare > best)
            {
                best = compare;
                match = loader;
            }
        }

        return match;
   }
}

void digCommonQBert ()
{
    void *foo = &digCommonImageLoader_png.read_data;
}

private import net.BurtonRadons.dig.platform.imageLoaderPNG;
private import net.BurtonRadons.dig.platform.imageLoaderJPEG;
private import net.BurtonRadons.dig.common.imageLoaderTGA;

static this ()
{
    new digCommonImageLoader_png ();
    new digCommonImageLoader_jpeg ();
    new digCommonImageLoader_tga ();
}
